<?php

namespace App\Http\Controllers\Admin;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Lib\FormProcessor;
use App\Models\ApiProvider;
use App\Models\BillCategory;
use App\Models\Form;
use App\Models\Company;
use App\Models\TransactionCharge;
use App\Rules\FileTypeValidate;
use Exception;
use Illuminate\Http\Request;

class CompanyController extends Controller
{
    public function all()
    {
        $pageTitle  = 'Company List';
        $baseQuery  = Company::searchable(['name'])->with('form')->orderBy('id', getOrderBy());
        $categories = BillCategory::active()->get();

        if (request()->export) {
            return exportData($baseQuery, request()->export, "Company");
        }

        $companies = $baseQuery->paginate(getPaginate());
        $charge    = TransactionCharge::where('slug', "utility_charge")->first();


        return view('admin.company.all', compact('pageTitle', 'companies', 'categories', 'charge'));
    }

    public function save(Request $request, $id = 0)
    {
        $imageValidation = $id ? 'nullable' : 'required';

        $request->validate([
            'name'           => 'required|max:255|unique:companies,name,' . $id,
            'fixed_charge'   => 'nullable|numeric|min:0',
            'category_id'    => 'required|numeric|exists:bill_categories,id',
            'percent_charge' => 'nullable|numeric|min:0',
            'minimum_amount' => 'nullable|numeric|min:0',
            'maximum_amount' => 'nullable|numeric|min:0|gt:minimum_amount',
            'image'          => [$imageValidation, 'image', new FileTypeValidate(['jpg', 'jpeg', 'png'])],
        ]);

        if ($id) {
            $utility  = Company::findOrFail($id);
            $notify[] = ['success', 'Utility bill setting updated successfully'];
        } else {
            $utility  = new Company();
            $notify[] = ['success', 'Utility bill setting added successfully'];
        }

        if ($request->hasFile('image')) {
            try {
                $old            = $utility->image;
                $utility->image = fileUploader($request->image, getFilePath('utility'), getFileSize('utility'), $old);
            } catch (\Exception $exp) {
                $notify[] = ['error', 'Couldn\'t upload your image'];
                return back()->withNotify($notify);
            }
        }

        $utility->name           = $request->name;
        $utility->fixed_charge   = $request->fixed_charge;
        $utility->percent_charge = $request->percent_charge;
        $utility->category_id    = $request->category_id;
        $utility->minimum_amount = $request->minimum_amount ?? null;
        $utility->maximum_amount = $request->maximum_amount ?? null;
        $utility->save();

        return back()->withNotify($notify);
    }

    public function configure($id)
    {
        $utility   = Company::findOrFail($id);
        $pageTitle = 'Configure Utility Bill - ' . $utility->name;
        $form      = Form::where('act', 'utility_biller')->where('id', $utility->form_id)->first();
        return view('admin.company.form', compact('pageTitle', 'utility', 'form'));
    }

    public function saveConfigure(Request $request, $id)
    {
        $setup               = Company::findOrFail($id);
        $formProcessor       = new FormProcessor();
        $generatorValidation = $formProcessor->generatorValidation();

        $request->validate($generatorValidation['rules'], $generatorValidation['messages']);

        $exist         = Form::where('act', 'utility_biller')->where('id', $setup->form_id)->first();
        $generatedForm = $formProcessor->generate('utility_biller', $exist, 'act', $setup->form_id);

        $setup->form_id = @$generatedForm->id ?? 0;
        $setup->save();

        $notify[] = ['success', 'Utility bill setting has been successfully configured'];
        return back()->withNotify($notify);
    }

    public function status($id)
    {
        return Company::changeStatus($id);
    }

    public function fetchCompanies()
    {

        $provider = ApiProvider::whereJsonContains('module', 'utility_bill')->where('status->utility_bill', Status::YES)->first();

        if (!$provider) {
            $notify[] = ['error', 'At least one automated utility bill provider integration is required to import biller categories.'];
            return back()->withNotify($notify);
        }

        $pageTitle            = 'Supported Companies';
        $existingCompanyNames = Company::pluck('name')->toArray();

        try {
            $className  = 'App\\Lib\\UtilityBillProvider\\' . $provider->alias;
            $apiBillers = (new $className())->getBillers();
        } catch (Exception $ex) {
            $notify[] = ['error', $ex->getMessage()];
            return back()->withNotify($notify);
        }

        $billers = collect($apiBillers->content ?? []);
        session()->put('billers', $billers);

        return view('admin.utility_bills.fetch_billers', compact('pageTitle', 'billers', 'existingCompanyNames'));
    }

    public function saveCompanies(Request $request)
    {
        $request->validate([
            'biller_ids' => 'required|array|min:1',
        ]);

        $billerArray    = [];
        $sessionBillers = collect(session('billers'))->keyBy('id');

        $currencyText = strtoupper(gs('cur_text'));

        session()->forget('billers');

        foreach ($request->biller_ids as $billerId) {

            $item = $sessionBillers->get($billerId);

            if (!$item || Company::where('name', $item->name)->exists()) continue;

            $companyCurrency = $item->localTransactionCurrencyCode ?? 'N/A';

            if ($currencyText != strtoupper($companyCurrency)) {
                $notify[] = ['error', "The company[$item->name] currency not match the your system currency. so please unselect this company"];
                return back()->withNotify($notify);
            }

            $category = BillCategory::where('name', $item->type)->first();

            if (!$category) {
                $category             = new BillCategory();
                $category->name       = $item->type;
                $category->created_at = now();
                $category->updated_at = now();
                $category->save();
            }

            $billerArray[] = [
                'name'           => $item->name,
                'category_id'    => $category->id,
                'minimum_amount' => $item->minLocalTransactionAmount ?? null,
                'maximum_amount' => $item->maxLocalTransactionAmount ?? null,
                'fixed_charge'   => $item->internationalTransactionFee ?? null,
                'percent_charge' => $item->internationalTransactionFeePercentage ?? null,
                'company_data'   => json_encode($item),
                'created_at'     => now(),
                'updated_at'     => now(),
            ];
        }

        Company::insert($billerArray);

        $notify[] = ['success', 'The ' . count($billerArray) . ' billers company added successfully'];
        return to_route('admin.utility.bill.company.all')->withNotify($notify);
    }
}
